/*
 * cmfgetln.c
 *  
 *
 * Aternatives:
 * 1.  fgets
 *		Pro:  Ubiquitous
 *		Con:  Buffer size issues
 * 2.  C++ iostream getline
 *		Pro:  Buffer OK using C++ string
 *		Con:  Slow, requires iostream and C++
 * 3.  fgetln/getline
 *		Pro:  OK buffer, speed swell, uses FILE *, char * types
 *		Con:  BSD fgetln not on GNU systems, vice versa
 *
 */
#include <sys/param.h> 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cmfgetln.h"
#include "common.h"


static unsigned short int Bfs = 40;

/*
 * cmfgetln is the easy-to-use version of cmfgettoch for 
 * getting a new line quick.  See notes before cmfgettoch.
 *
 * (Should I be a macro?)
 */
char *cmfgetln(FILE *ifp)
{

    return cmfgettoch(ifp, '\n');
}


/*
 * cmfgettoch will read from the FILE pointer ifp until the character to is found, and will 
 * return a string of all the data thereto, replacing the character to with a \0 character.  
 * Will return NULL if called during an EOF condition.
 * 
 * NOTE:  Buffers allocated here should be freed by whatever function uses it.
 *
 * This function should probably be made far more efficient.  I used to have a brilliant 
 * (or what I thought was brilliant) optimizationf for BSD, but it seemed to act wacky on 
 * NetBSD sometimes.  I saw some promising replacement code in gencat, using read, 
 * but haven't yet given it a go of replacing this.
 */
char *cmfgettoch(FILE *ifp, const char to)
{
    register int ch;
    register char *buf;
    register size_t blen, cnt;
    
    
    cnt = 0;
    blen = Bfs;
    buf = (char *)xmalloc(blen);
    
    while ((ch = fgetc(ifp)) != to) {
        buf[cnt] = ch;
        
        /* whoa...end of file.  Time to stop */
        if (ch == EOF)
            break;
            
        cnt++;
        
        /* out of buffer space...reallocate */
        if (cnt >= blen) {
            blen += Bfs;
            buf = (char *)xrealloc(buf, blen);
        }
    }
    
    /* Return NULL terminated string */
    buf[cnt] = '\0';
    
    /* No more lines...return NULL pointer */
    if ((cnt == 0) && (ch == EOF)) {
        free(buf);
        return NULL;
    }
    
    /* All is OK, return the string */
    return buf;
}

/*
 * cmtunebfs will set how large of a buffer to reallocate every time a new 
 * buffer is created or needs to be reallocated.  By varying this number, 
 * one can carefully adjust ones needs of processor usage via realloc versus 
 * memory wastage by malloc-ing too much RAM.
 *
 * Although cmtunebfs may reject the value passed to it, it won't return 
 * an error condition, it will continue working, but with the previous value 
 * therein.
 *
 * We do reject all values less than 16, most of these really hurt performance.  
 * It's suggested that immensely huge values are not used, so we cut those off at 
 * 256.  If you find a circumstance in which these values are daft, well, feel 
 * free to edit this.
 */
void cmtunebfs(const unsigned short int newbfs)
{

    if (newbfs < 16)
        return;
    
    if (newbfs > 256) {
        Bfs = 256;
        return;
    }
    
    Bfs = newbfs;
}