/*
 * parse.c
 * The taxman's taken all my dough.
 *
 * Created by mercadal on Fri Feb 01 2002.
 *
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cmfgetln.h"
#include "common.h"
#include "imagehandler.h"
#include "linktitlehandler.h"


#define NUMDEFTAGS 50


static void blankHandler(char *);
static void defaultHandler(char *);
static void readUdefs(FILE *);

void initializeTags(FILE *, char);
void parseTag(void);
void uninitializeTags(void);


struct ftnt {
    char *there;
    char *repl;
};

typedef struct {
    struct ftnt *myTag;
    void (*handler)(char *);
} mtag;


static char quiet;
static int numtags = NUMDEFTAGS;	/* default set has NUMDEFTAGS mtags */
static int numudefs = 0;
static struct ftnt **udefs;
static mtag *knowTags;


static struct ftnt deftags[NUMDEFTAGS] = {
    {"title", ""},	/* keep at position 0 */
    {"link", "<strong>"},	/* keep at position 1 */
    {"url", ""},	/* keep at 2 */
    {"height", ""},	/* keep at 3 */
    {"width", ""},	/* keep at 4 */
    {"/image", "</div>"},	/* keep at 5 */
    
    {"language", ""},	/* position 6-12 are blank handler */
    {"docs", ""},
    {"ttl", ""},
    {"skipHours", ""},
    {"skipDays", ""},
    {"generator", ""},
    {"guid", ""},
    
    {"/language", ""},
    {"/docs", ""},
    {"/ttl", ""},
    {"/skipHours", ""},
    {"/skipDays", ""},
    {"/generator", ""},
    
    {"image", "<div style=\"float: right\">"},
    {"/width", ""},
    {"/height", ""},
    {"/url", ""},
    {"/guid", ""},
    
    {"/title", ""},
    {"/link", "</strong>"},
    
    {"channel", ""},
    {"/channel", "<br>"},
    
    {"item", ""},
    {"/item", "<br>"},
    
    {"description", "<br><em>"},
    {"/description", "</em><br>"},
    
    {"copyright", "<!--RSS feed copyright:  "},
    {"/copyright", " -->"},
    
    {"managingEditor", "<!--Editorial contact for RSS feed:  "},
    {"/managingEditor", "-->"},
    
    {"webMaster", "<!--Technical contact for RSS feed:  "},
    {"/webMaster", "-->"},
    
    {"pubDate", "<span style=\"font-size: smaller\">Published:  <code>"},
    {"/pubDate", "</code></span><br><hr>"},
    
    {"lastBuildDate", "<span style=\"font-size: smaller\">Last Changed:  <code>"},
    {"/lastBuildDate", "</code></span><br><hr>"},
    
    {"category", "<!--RSS feed category:  "},
    {"/category", "-->"},
    
    {"author", "<span style=\"font-size: smaller\">Author:  <code>"},
    {"/author", "</code></span><br>"},
    
    {"author", "<span style=\"font-size: smaller\">Comments page:  <code>"},
    {"/author", "</code></span><br>"},
    
    {"*RDFSTART*", ""},	/* keep second from the last */
    {"*RDFEND*", ""}	/* keep the last */
};


/*
 * Let's initialize these mtags!
 */
void initializeTags(FILE *udeffile, char runquiet)
{
    register int cnt;
    
    
    /* should we run quiet? */
    quiet = runquiet;
    
    /* 
     * Our user didn't specify their own user-defined tags to 
     * run in the parser.  As such, we'll use the default 
     * rss2html set. 
     */
    knowTags = (mtag *)xmalloc(numtags * sizeof(mtag));

    for (cnt = 0; cnt < numtags; cnt++) {
        knowTags[cnt].myTag = &deftags[cnt];
        
        switch (cnt) {
            case 0: /* title case, need title handler */
                knowTags[cnt].handler = titleHandler;
                break;
            case 1: /* link case */
                knowTags[cnt].handler = linkHandler;
                break;
            case 2:
                knowTags[cnt].handler = urlHandler;
                break;
            case 3:
                knowTags[cnt].handler = heightHandler;
                break;
            case 4:
                knowTags[cnt].handler = widthHandler;
                break;
            case 5:
                knowTags[cnt].handler = endImageHandler;
                break;
            case 6: case 7: case 8: case 9:	case 10: 
            case 11: case 12: /* blank cases */
                knowTags[cnt].handler = blankHandler;
                break;
            default:
                knowTags[cnt].handler = defaultHandler;
                break;
        }
    }
        
    if (udeffile != NULL)
        readUdefs(udeffile);
    
    /* Let's do the *RDFSTART* stuff */
    knowTags[NUMDEFTAGS - 2].handler(knowTags[NUMDEFTAGS - 2].myTag->repl);
}


/*
 * The parser for the so-called user defined tags file 
 * is pretty poor and could probably do with some gussying
 * up by the more intrepid coder.  Namely, we realloc for each 
 * line which is not necessarily efficient.
 */
void readUdefs(FILE *from)
{
    char *line, *tmprepl;
    register char reassigned, useblank;
    register int cnt, index;
    struct ftnt *tmpudef;
    

    udefs = (struct ftnt **)xmalloc(1 * sizeof(struct ftnt *));
    
    /* 
     * Can I do this more efficiently? 
     * Say 16 bit Intel assembly and I kill you.  :-)
     */
    cnt = 0;
    useblank = 0;
    
    while ((line = cmfgetln(from)) != NULL) {
        if ((isspace(*line)) || (*line == '\0') || (*line == '#'))	/* comments */
            continue;
            
        /*
         * Read in tag into udef[cnt].  If there's a replacement string, store it.
         * Otherwise, assume a blank handler.
         */
        udefs[cnt] = tmpudef = (struct ftnt *)malloc(sizeof(struct ftnt));
        
        tmpudef->there = xstrdup(strtok(line, "\\"));
        
        if ((tmprepl = strtok(NULL, "\\")) != NULL)
            tmpudef->repl = xstrdup(tmprepl);
        else
            useblank = 1;
        
        /*
         * Search through known tags.  If this udef contains a known tag, replace 
         * that with the udef.
         */
        for (index = 0, reassigned = 0; index < numtags; index++) {
            if (strcmp(tmpudef->there, knowTags[index].myTag->there) == NULL) {
                knowTags[index].myTag = tmpudef;
                reassigned = 1;
                
                if (useblank)
                    knowTags[index].handler = blankHandler;
                
                break;
            }
        }
        
        /* ...or just add it if unknown as of yet */
        if (!reassigned) {
            numtags++;
            knowTags = (mtag *)realloc(knowTags, numtags * sizeof(mtag));
            knowTags[numtags - 1].myTag = tmpudef;
            
            if (useblank)
                knowTags[numtags - 1].handler = blankHandler;
            else
                knowTags[numtags - 1].handler = defaultHandler;
        }
        
        cnt++;
        useblank = 0;
        udefs = (struct ftnt **)xrealloc(udefs, (cnt + 1) * sizeof(struct ftnt *));
        free(line);
    }
    
    numudefs = cnt;
}


void uninitializeTags(void)
{
    register int cnt;
    
    
    /* Do *RDFEND* stuff */
    knowTags[NUMDEFTAGS - 1].handler(knowTags[NUMDEFTAGS - 1].myTag->repl);
    
    for (cnt = 0; cnt < numudefs; cnt++) {
        free(udefs[cnt]->there);
        free(udefs[cnt]->repl);
        free(udefs[cnt]);
    }
    
    free(udefs);
    free(knowTags);
}


void parseTag(void)
{
    char *buf;
    register int cnt;


    buf = cmfgettoch(stdin, '>');
    
    for (cnt = 0; cnt < numtags; cnt++) {
        if (strcmp(knowTags[cnt].myTag->there, buf) == NULL) {
            free(buf);
            
            /* Run this tag's handler */
            knowTags[cnt].handler(knowTags[cnt].myTag->repl);
            return;
        }
    }
    
    /* Unknown tag */
    if (!quiet)
        (void)fprintf(stderr, "Don't know %s tag\n", buf);
    
    free(buf);
}


void defaultHandler(char *print)
{

    (void)printf("%s", print);
}


void blankHandler(char *print)
{
    char *discard;
    
    discard = cmfgettoch(stdin, '<');
    free(discard);
    
    parseTag();
}
